#
############################################################
# MODULE:    Tag-valid SGML to MARC record converter
# VERSION:   1.0
# DATE:      November 17, 1997
#
# MULBERRY INTERNAL VERSION CONTROL:
# $Id: sgm2mrc.pl,v 1.12 1997/11/26 15:11:50 tkg Exp $
############################################################

############################################################
# SYSTEM:    MARC to tag-valid SGML converter set
#
# PURPOSE:   Convert tag-valid SGML to MARC records
#
# CONTAINS:  1) "Use" statements for external packages
#            2) Declarations of constants
#            3) Declarations of global variables
#            4) Command-line argument processing
#            5) Initialization of log file and output file
#            6) MARC Description File processing
#            7) Declaration of handlers for processing input file
#            8) Input file processing
#            9) End processing
#            10) Subroutines
#
# PACKAGES REQUIRED:
#            1) David Megginson's SGMLS and SGMLS::Output packages
#            2) sgmlspl package
#            3) marcconv.pl MARC Description File handler
#
# CREATED FOR:
#            Network Development and MARC Standards Office
#            The Library of Congress
#
# ORIGINAL CREATION DATE:
#            November 1997
#
# CREATED BY:
#            Mulberry Technologies, Inc.
#            17 West Jefferson Street, Suite 207
#            Rockville, MD  20850
#            Phone:  301/315-9631
#            Fax:    301/315-8285
#            e-mail: info@mulberrytech.com
#            WWW:    http://www.mulberrytech.com
############################################################



############################################################
#                   DESIGN CONSIDERATIONS
############################################################


############################################################
# External packages

# David Megginson's SGMLpm package
use SGMLS;
use SGMLS::Output;

# Object-oriented version of David Megginson's SGMLSpl.pl
use SGMLSpl;

# Routines for handling MARC Description Files
require "Marcconv/marcdesc.pl";

############################################################
# Constants

# Maximum length of a MARC record, largely because we have space for
# five digits when we put the length in the Leader
$cMarcMaxRecordLength = 99999;

# Magic strings to indicate type of control field
$cNoIndicatorsOrSubfields = 'noindsf';
$cPositionallyDefinedField = 'posdef';

# Constant strings for significant MARC characters
#
# End of record delimiter
$cMarcEOR = "\035";
# End of field delimiter
$cMarcEOF = "\036";
# End of subfield delimiter
$cMarcEOS = "\037";
# Blank character
$cMarcBlank = ' ';
# Blank symbol used in MARC Description File
$cMarcBlankSymbol = '#';
# Fill character
$cMarcFill = "|";

# Magic string to indicate what to use as key for selecting clusters
# for positionally-defined field
$cLeader0607Key = 'LEADER0607';

# Where to find the MARC Description File DTD
$cMarcDescDtd = &FindMarcConvFile('marcdesc.dtd');
# Where to find the MARC Description File
$cMarcDescriptionFile = &FindMarcConvFile('marcdesc.sgm');

# Constants for character conversion
# Magic strings to both determine conversion type and to output
# to log file to indicate conversion type
$cSGMLConversion = 'SGML';
$cRegisterConversion = 'Upper register to lower register';
$cCharacterConversion = 'Character conversion';
$cUserDefinedConversion = 'User-defined';

# Where to find the default conversion specification file for each
# type of conversion
#
# The DTD for character-entity conversion files
$cConversionDtdFile = &FindMarcConvFile('entmap.dtd');
# Specification file for upper-register to entity conversion
$cRegisterConversionFile = &FindMarcConvFile('register.sgm');
# Specification file for non-ASCII character to entity conversion
$cCharacterConversionFile = &FindMarcConvFile('charconv.sgm');

# Usage message
# This is output if the user gets the command parameters wrong
$cUsage = <<EndOfUsage;
sgm2mrc.pl [-command file]
   [-sgmlconv | -registerconv | -charconv | -userconv file]
   [-log file] [-o file] [-marcdesc file] [-help] input-file
EndOfUsage

# Help message
# This is output if the user specifies the -help parameter
$cHelp = <<EndOfHelp;
sgm2mrc.pl [-command file]
   [-sgmlconv | -registerconv | -charconv | -userconv file]
   [-log file] [-o file] [-marcdesc file] [-help] input-file

where:
-command file
        Read program command options from "file".

-sgmlconv
        Perform minimal, "SGML sanity" character conversion using the
        built-in conversion table

-registerconv
        Convert upper-register characters to lower-register characters
        using the built-in conversion table.

        The minimal SGML conversion will also be performed.

-charconv
        Convert characters to entities using the built-in conversion
        table

        The minimal SGML conversion will also be performed.

-userconv file
        Perform character conversion using the user-supplied
        conversion specification in "file"

        An error will be signaled if "file" is not specified or if
        "file" cannot be opened, or if "file" is not a file of the
        correct format.

        The minimal SGML conversion will also be performed.

-log file
        Write the output log to "file".  If this option is not
        specified, the log will be written to "sgm2mrc.log" in the
        current directory.

-o file
        Write the unvalidated SGML output to "file" instead of to
        the default file "stdout.mrc".

-marcdesc file
        Read the MARC Description File named "file" instead of the
        default MARC Description File that the program automatically
        reads on initialization.

-help
        Print this help information then quit.

input-file
        The name of the input MARC record file
EndOfHelp

############################################################
# Global variables

# SGML file with "record type" data
$gMarcDescFile = $cMarcDescriptionFile;

# Log file name
$gLogFile = 'sgm2mrc.log';

# Count of records processed
$gRecordCount = 0;

# Count of records converted
$gConvertCount = 0;

# Count of records skipped
$gSkipCount = 0;

# Output file handle
$gOutputFile = 'stdout.mrc';

# Conversion specification count
$gConversionSpecCount = 0;

# Entity conversion map file
$gEntityConversionFile = '';

############################################################
# Process command-line arguments

# First check for a "-command  file" argument, find the command file,
# parse it for commands, then prepend the commands to the argument
# list in @ARGV
for($lArgCount = 0; $lArgCount <= $#ARGV; $lArgCount++) {

    if ($ARGV[$lArgCount] =~ /^-command$/) {
	local($lJunk, $lCommandFile) = splice(@ARGV, $lArgCount, 2);
	local($lCommandData, @lCommandData) = ();

#	print STDERR ":$lJunk:$lCommandFile:\n";

	open(COMMANDFILE, "$lCommandFile") ||
	    die "Couldn't open \"$lCommandFile\" as command file.\n";

	# We want the command file name later
	$gCommandFile = $lCommandFile;

	while (<COMMANDFILE>) {

	    # Allow comments in lines beginning with "#"
	    next if /^#/;

	    # Tidy up the line before we split on whitespace
	    chomp;
	    s/^\s+//;
	    s/\s+$//;

	    push(@lCommandData, split(/\s+/));
	}

	# Put the arguments from the command file *before* the command line
	# arguments
	unshift(@ARGV, @lCommandData);

	last;
    }
}
	
# Process the command line (and command file) arguments
while (@ARGV) {
    if ($ARGV[0] =~ /^-/) {
	if ($ARGV[0] =~ /^-command$/) {
	    warn "\"-command\" argument may only be used once.\n";
	    die $cUsage;
	} elsif ($ARGV[0] =~ /^-marcdesc$/) {
	    shift;
	    $gMarcDescFile = shift;
	} elsif ($ARGV[0] =~ /^-log$/) {
	    shift;
	    $gLogFile = shift;
	} elsif ($ARGV[0] =~ /^-o$/) {
	    shift;
	    $gOutputFile = shift;
	} elsif ($ARGV[0] =~ /^-sgmlconv$/) {
	    shift;
	    $gConversionType = $cSGMLConversion;
	    $gConversionSpecFile = '';
	    $gConversionSpecCount++;
	} elsif ($ARGV[0] =~ /^-registerconv$/) {
	    shift;
	    $gConversionType = $cRegisterConversion;
	    $gConversionSpecFile = $cRegisterConversionFile;
	    $gConversionSpecCount++;
	} elsif ($ARGV[0] =~ /^-charconv$/) {
	    shift;
	    $gConversionType = $cCharacterConversion;
	    $gConversionSpecFile = $cCharacterConversionFile;
	    $gConversionSpecCount++;
	} elsif ($ARGV[0] =~ /^-userconv$/) {
	    shift;
	    $gConversionSpecFile = shift;
	    $gConversionType = $cUserDefinedConversion;
	    $gConversionSpecCount++;
	} elsif ($ARGV[0] =~ /^-help$/) {
	    die $cHelp;
	} else {
	    warn "Unknown option \"$ARGV[0]\".\n\n";
	    die $cUsage;
	}
    } else {
	last;
    }
}

############################################################
# Open log file before we go any further

open(LOGFILE, ">$gLogFile") ||
    warn "Couldn't open \"$gLogFile\" as log file." .
        " Continuing without log.\n";

&LogOpenMessage();

if (!@ARGV) {
    warn "An input file must be specified.\n\n";
    die $cUsage;
} elsif (@ARGV > 1) {
    warn "Only one input file can be specified." .
	"  \"" . join(" ", @ARGV) . "\" is too many file names.\n\n";
    die $cUsage;
} else {
    $gInputFile = shift;
}

############################################################
# Open output file

if ($gOutputFile ne '') {
    open(OUTFILE, ">$gOutputFile") ||
	die "Couldn't open \"$gOutputFile\" as output file.\n";
} else {
    *OUTFILE = STDOUT;

    $gOutputFile = "STDOUT";
}


############################################################
# Work out the conversion type and evaluate the conversion
# specification file
if ($gConversionSpecCount > 1) {
    warn "Only one conversion specification is allowed.\n\n";
    die $cUsage;
} elsif ($gConversionSpecCount == 0) {
    $gConversionType = $cSGMLConversion;
} elsif ($gConversionType eq $cRegisterConversion ||
	 $gConversionType eq $cCharacterConversion ||
	 $gConversionType eq $cUserDefinedConversion) {
    &EvalConversion($gConversionSpecFile);
}

############################################################
# Write what we know about the processing to the log file
$gCommandLineArguments = <<EndOfArguments;
------------------------------------------------------------
Command File:          $gCommandFile
MARC Description File: $gMarcDescFile
Input File:            $gInputFile
Output File:           $gOutputFile
Character Conversion:  $gConversionType
Conversion File:       $gConversionSpecFile
Log File:              $gLogFile
------------------------------------------------------------
EndOfArguments

&Log($gCommandLineArguments);

############################################################
# Process the MARC Description File

open (MARCDESC, "nsgmls $cMarcDescDtd $gMarcDescFile |");
SGMLSPL::Process($gMarcDesc, MARCDESC);

# Uncomment this subroutine call to dump the data structures for the
# MARC Description File to STDOUT
# &DumpMarcDesc();

############################################################
# Declare handlers for processing the input SGML
#
# The input SGML is processed using the facilities provided by David
# Megginson's sgmlspl package, except that the package has been
# generalized to support multiple instances of SGML processing because
# we also use it for the MARC Description File and possibly the
# entity-character conversion specification file.
#
# In this instance, the processing is implemented as a state machine.
# The sgmlspl package supports handlers for non-specific
# "start_element" and "end_element" events in addition to the ability
# to declare handlers for start and end tags of specific elements.
# The generic handlers are used because of the large number of
# elements in the MARC DTDs and because the elements are not known
# beforehand: if element-specific handlers were used, the MARC
# Description File would have to be processed to generate every
# possible element name, and two subroutines for every element (its
# start and end handlers) would have to be generated then evaluated
# (in the same manner that the entity-character conversion file
# results in a subroutine being constructed then evaluated, but there
# would be so many more subroutines for every possible element in the
# tag-valid SGML file).  Element-specific handlers would significantly
# increase the startup time and slow the running of the program.
#
# The generic start_element and end_element handlers, only, are used,
# but the subroutines run by the start_element and end_element
# handlers are set each time the processing state changes.  For
# example, processing starts in the "StartProcessing" state, then if
# the element name matches that of a group tag, the state changes to
# the "Record" state, but if the element name matches neither a group
# tag or a wrapping tag, the state changes to the "Error" state.
#
# When the state changes, the start_element and/or the end_element
# handlers are changed to use the subroutine corresponding to the new
# state.  The start_element and end_element handlers are not always
# set to the same state, firstly because many of the elements in the
# MARC DTDs are EMPTY and don't have end tags (even though nsgmls will
# output them because it doesn't know any better and standard ESIS
# includes EMPTY tag end tags anyway), and secondly because the
# occurrence of the end tag is the thing that puts you into that
# state.
#
# The states are:
#
# Error
#    An error has occurred.  Processing enters the "Error" state when
#    the current tag doesn't match any of the expected patterns, and
#    stays in the "Error" state until an element group start tag is
#    recognized.
#
# Generic
#    A "do nothing" state.  For example, the end_element handler may
#    be set to the "Generic" state when the significant events occur
#    at the start elements and nothing is currently required to happen
#    when we see an end element.
#
# NoIndSf
#    The current element is for a MARC field without indicators or
#    subfields (i.e. it is a control variable field, but is not
#    positionally-defined).  Since these elements contain character
#    data, all the action happens when we see the end tag.
#
# PosDef
#    The current element is for a positionally-defined field.  The
#    element may be the containing element or may be an element for
#    one of the clusters within the positionally-defined field.
#
# ProcessGroups
#    The current element is a group start tag or the next significant
#    event occurs when we see a group end tag.
#
# ProcessLeader
#    The current element is the Leader wrapper tag or one of the
#    elements for the clusters within the Leader.
#
# ProcessMarcTag
#    The current element is the containing element for a data variable
#    field or is an element for a subfield of a data variable field.
#
# Record
#    The current element is the wrapper element for a complete MARC
#    record.  For the start element, either the next element is for
#    the Leader or it is an error.  For the element end, the MARC
#    record is built from the data that has been accumulated while
#    processing the tag-valid SGML for the record.
#
# StartProcessing
#    The initial state, which is reentered at the end of element
#    groups and at the end of processing the tag-valid SGML for a
#    complete MARC record.
#
# VariableData
#    The current element is for a variable data field.

# Create a new SGMLSPL object for the input tag-valid SGML
$gInputSGML = new SGMLSPL;

# "StartProcessing" state start_element handler.
# Initialize the hash for the data in the MARC record, increment the
# count of records seen, then change state to the "Record" state if
# the current element is an element group, or change to the "Error"
# state if we don't recognise the current element.
sub StartProcessing_Start {
    my $lElement = shift;

    # Save the output
    push_output 'string';

    # All element name handling is with lowercase element names.
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "StartProcessing_Start:<" . $lElement->name . ">";

#    print ":" . $lName . ":" . $gDtdGroups{$lName} . ":\n";

    # Initialize the hash containing the MARC record data
    undef(%gMarcData);

    # Increment the count of records seen
    $gRecordCount++;

    # If the current element is a top-level tag for a MARC record, it will
    # have element groups defined for it in the MARC Description File, and
    # we can continue, but if it's not in the MARC Description File and we
    # are not still at the outer, multiple-record wrapper level, then it's
    # an error.
    if (defined($gDtdGroups{$lName})) {

	# We will use the document type later
	$gDoctype = $lName;

	# The next state is "Record", and the action occurs in the start
	# tag
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Record_Start);
    } elsif ($lName =~ /mrc.?file/i) {
	# Do nothing if we are still at the wrapper element
    } else {
	# To get here, we had an element name that we didn't know what
	# to do with.

	# Increment the count of records skipped because of errors
	$gSkipCount++;

	# Write a log message for posterity
	&Log("Record $gRecordCount has unknown top-level tag \"$lName\";  record not converted.");

	# Go to "Error"
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);
    }
}

# "Record" state start_element handler
# To get here, we found a top-level tag that we recognized.  We already
# handled that tag, but the next tag we expect to see is the start tag
# for the Leader wrapper.  Before we look at what the current tag is, we
# setup the hash for the tag ranges for our current document type and
# we pre-fill the Leader data with blanks because anything that doesn't
# get filled from the SGML should be a blank when we make the MARC record.
sub Record_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

    # The hash for element groups doesn't use the most useful key for
    # our present purpose, so we make a new hash using the label as
    # the key since that's what we see first.
    foreach (keys(%{$gDtdGroups{$gDoctype}})) {

	$gTagRange{${$gDtdGroups{$gDoctype}}{$_}->label} =
	    [ ${$gDtdGroups{$gDoctype}}{$_}->start,
	    ${$gDtdGroups{$gDoctype}}{$_}->end ];

#	print STDERR ":" . ${$gDtdGroups{$gDoctype}}{$_}->label . ":" .
#	    ${$gDtdGroups{$gDoctype}}{$_}->start . ":" .
#	    ${$gDtdGroups{$gDoctype}}{$_}->end . ":\n";
    }

    $gLeaderData = $cMarcBlank x 24;

#    print "Record_Start:<" . $lElement->name . ">\n";

    # If the present tag is the Leader wrapper, change the state to
    # "ProcessLeader", otherwise it's an error and we change state to
    # "Error".
    if ($lName =~ /^${gDoctype}ldr-..$/) {
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&ProcessLeader_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&ProcessLeader_End);
    } else {

	# Increment the count of records skipped
	$gSkipCount++;

	# Record this moment for posterity
	&Log("Record $gRecordCount has unknown Leader element \"$lName\";  record not converted.");

	# Go to "Error" and stay there until it is safe to come out
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);
    }
}

# "Record" state end_element handler
# If we got this far, then we have the data for a complete MARC record, so
# we call &BuildMarcRecord to turn the data structures we've been collecting
# into a MARC record.
sub Record_End {
    my $lElement = shift;

#    print "Record_End:</" . $lElement->name . ">\n";

#    print ":Leader:$gLeaderData:\n";

#    foreach $lMarcTag (sort(keys(%gMarcData))) {
#	print ":$lMarcTag:$gMarcData{$lMarcTag}:\n";
#    }

    # This is where the work gets done.
    &BuildMarcRecord();

    # And it's back to the beginning again...
    SGMLSPL::sgml($gInputSGML,
		      'start_element',
		      \&StartProcessing_Start);

    SGMLSPL::sgml($gInputSGML,
		      'end_element',
		      \&Generic_End);
}

# "ProcessLeader" state start_element handler
# To get here, we've seen the start tag for the wrapper element for
# the Leader, so we expect to see a sequence of Leader cluster elements.
# If we don't see the right type of element, it's an error (of course),
# and if we do, we extract the data in the "value" attribute and insert
# it into the right place in the Leader string.
sub ProcessLeader_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "ProcessLeader_Start:<" . $lElement->name . ">\n";

    # If the current tag matches the pattern for a Leader cluster, then
    # extract the "value" attribute, make it nice, and insert it into
    # the Leader string.
    if ($lName =~ /^${gDoctype}ldr-..-(\d{2})(-(\d{2}))?$/) {
	# We don't always get a $lEndChar
        local($lStartChar, $lEndChar) = ($1, $3);
	local($lFieldWidth) = 0;
	local($lAttributeValue) = '';

#        print ":$lStartChar:$lEndChar:";

	# If the cluster is not a range, the end character number is
	# the same as the start number
	if ($lEndChar eq '') {
	    $lEndChar = $lStartChar;
	}

	$lFieldWidth = $lEndChar - $lStartChar + 1;

#        print "$lStartChar:$lEndChar:$lFieldWidth:\n";

	# Get the attribute value
	$lAttributeValue = &NiceAttribute($lElement->attribute(VALUE)->value);

	# If we are a blank or fill character, expand to the field width
	$lAttributeValue = $lAttributeValue x $lFieldWidth
	    if ($lAttributeValue eq $cMarcBlank ||
		$lAttributeValue eq $cMarcFill);

	# Put it where it belongs
	substr($gLeaderData, $lStartChar, $lFieldWidth) =
	    sprintf "%${lFieldWidth}s", $lAttributeValue;

    } else {
	# To get here, we didn't recognise the tag as being a Leader
	# cluster

	$gSkipCount++;

	# Save this moment for posterity
	&Log("Record $gRecordCount has unknown Leader cluster element \"$lName\";  record not converted.");

	# Go to "Error", go directly to "Error", do not pass "StartProcessing"
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Error_End);
    }
}

# "ProcessLeader" state end_element handler
# To get here, we've seen a Leader wrapper element start tag.  nsgmls will
# output ESIS events for Leader cluster tags even though the cluster elements
# are EMPTY, but the only end tag that we are interested in is the end
# tag for the Leader wrapper element.  Once we've found that, we know we
# have all of the Leader, so we can change state to "ProcessGroups", and
# we can also find out what's in Leader character positions 06 and 07
# because we may need that later.
sub ProcessLeader_End {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "ProcessLeader_End:</" . $lElement->name . ">\n";

    if ($lName =~ /^${gDoctype}ldr-..$/) {

        # Leader characters 06 and 07 are reused in several places
        $gLeader0607 = substr($gLeaderData, 6, 2);
	# But when we use it, we want the blank symbol, "#", not the character
	$gLeader0607 =~ s/$cMarcBlank/$cMarcBlankSymbol/go;

	# The only thing left to do is change the state to "ProcessGroups"
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&ProcessGroups_Start);

	# But nothing significant happens with end tags while in
	# the "ProcessGroups" state
        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Generic_End);
    }
}

# "ProcessGroups" state start_element handler
# To get here, we've processed the Leader, and now we're expecting a
# sequence of element group tags, each of which will contain one or
# more elements for control or data variable fields.
sub ProcessGroups_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "ProcessGroups_Start:<" . $lElement->name . ">\n";

    # If we don't recognise the tag name as being one of our defined
    # element groups, then it's a problem.
    if (defined($gTagRange{$lName})) {

	# We worked this out earlier
	($gRangeStart, $gRangeEnd) = @{$gTagRange{$lName}};

	# As so often happens, sometimes ranges aren't
	if ($gRangeEnd eq '') {
	    $gRangeEnd = $gRangeStart;
	}

#	print ":$gRangeStart:$gRangeEnd:\n";

	# We really only wanted to check that we recognised the grouping
	# tag.  Now that we did, we can process any elements for MARC
	# fields that are within this grouping tag.
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&ProcessMarcTag_Start);

	# But we don't necessarily want to do much with the end tags until
	# we see the end of the element group
        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&ProcessGroups_End);
      } else {

	$gSkipCount++;

	# Didn't you just know that we wanted to say why it's an error?
	&Log("Record $gRecordCount has unknown group tag, \"$lName\";  record not converted.");

        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Error_End);
    }
}

# "ProcessGroups" state end_element handler
# To get here, we're processing the end tag of an element that should be
# either the end tag of a grouping element or the end tag of the element
# for the MARC record.  We may also get the spurious end tags for the
# empty elements within the grouping elements, so we're not calling what
# we don't understand an error.
sub ProcessGroups_End {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "ProcessGroups_End:</" . $lElement->name . ">\n";

    # If we are the end tag for a grouping element, then either the next
    # tag is the start tag for the next grouping element or its the end
    # tag for the whole tag-valid SGML version of the MARC record.
    if (defined($gTagRange{$lName})) {
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&ProcessGroups_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Record_End);
      } elsif (defined($gDtdGroups{$lName})) {
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&StartProcessing_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Generic_End);
      }	  
}

# "ProcessMarcTag" state start_element handler
# To get here, we've seen the start tag for an element group, so until
# we end the group, all of the start tags should be start tags (or the
# only tags) for the elements for the MARC fields.  If we do match
# the pattern for a field's start tag, we need to work out what type of
# field it is and take action then set the state accordingly.
sub ProcessMarcTag_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "ProcessMarcTag_Start:<" . $lElement->name . ">\n";

    if ($lName =~ /^.{4}(\d{3})(-(..?))?$/) {
	my $lMarcTag = $1;

	# Control fields are specified in the MARC Description File,
	# and when we processed that file, we listed them in this hash
	if (defined($gControlFields{$lMarcTag})) {

	    # Control fields are one of two types, and not surprisingly,
	    # we kept track of the type of each control field specified
	    # in the MARC Description File
	    if ($gControlFields{$lMarcTag}->field_type =~
		/$cPositionallyDefinedField/o) {
		$lFieldLength = 0;

#		print ":$lMarcTag is positionally defined.\n";

		# Even if it is positionally-defined, there's two options
		# for selecting how long the field will be.  If the key
		# for selecting clusters is the data in Leader 06 and 07,
		# then we can work it out now, otherwise we need to wait
		# until we process the 00 character position.
		if ($gControlFields{$lMarcTag}->key
		    =~ /^$cLeader0607Key$/o) {
		    $lFieldLength = $gControlFields{$lMarcTag}->
			clusters->{$gLeader0607}->field_length;
		}

		# Assuming we have the length, fill the data with fill
		# characters.
		$gPosDefData = $cMarcFill x $lFieldLength;

#		print ":$gPosDefData:\n";

		# So far we've seen the start tag of the wrapper element.
		# Now we need to process each of the elements for the clusters
	        SGMLSPL::sgml($gInputSGML,
				  'start_element',
				  \&PosDef_Start);

		# The clusters don't have end tags, but we'll catch the
		# end of the current wrapper element.
	        SGMLSPL::sgml($gInputSGML,
				  'end_element',
				  \&PosDef_End);

	    } else {
		# We're still a control field, but since we're not
		# positionally-defined, we must be a field without
		# indicators or subfields.

#		print ":$lMarcTag has no indicators or subfields.\n";

		# This element has character data content.  Save the
		# character data for processing when we see the end tag
		push_output 'string';

		# Set the end_element handler
	        SGMLSPL::sgml($gInputSGML,
				  'end_element',
				  \&NoIndSf_End);
	    }
	} else {
	    # To get here, we weren't a control field, therefore we are
	    # a data field (or we're terribly confused)

#	    print ":$lMarcTag is not a control field.\n";

	    # This start tag has two attributes for the two indicators.
	    # Get the attributes, make them civilised, and make them
	    # the first two characters of the data we're saving as the
	    # value of this field.
	    push(@{$gMarcData{$lMarcTag}},
		 (defined($lElement->attributes->{I1}) ?
		  &NiceIndicator($lElement->attribute(I1)->value) :
		  $cMarcBlank) .
		 (defined($lElement->attributes->{I2}) ?
		  &NiceIndicator($lElement->attribute(I2)->value) :
		  $cMarcBlank));

	    # We expect to be followed by a sequence of subfield elements
	    SGMLSPL::sgml($gInputSGML,
			      'start_element',
			      \&VariableData_Start);

	    SGMLSPL::sgml($gInputSGML,
			      'end_element',
			      \&VariableData_End);
	}
    }
}

# "PosDef" state start_element handler
# To get here, we've seen the start tag for the wrapper element for a set
# of positionally-defined field cluster elements.  If we don't match the
# pattern for a cluster element, it's an error.  If it is a cluster element,
# set the field length if we have to, then extract the data in the "value"
# attribute, make it civilised, and insert it into the string for the field.
sub PosDef_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "PosDef_Start:<" . $lElement->name . ">\n";

    if ($lName =~ /^${gDoctype}(\d{3})-..?-(\d{2})(-(\d{2}))?$/) {
        local($lMarcTag, $lStartChar, $lEndChar) = ($1, $2, $4);
	local($lFieldWidth) = 0;
	local($lAttributeValue) = '';

	# This is the other way of determining the length of a
	# positionally-defined field.  If we couldn't set the length
	# before, we can now get the 00 character position data to
	# use as the key for finding the field length.
	if ($lStartChar eq '00' &&
	    $gControlFields{$lMarcTag}->key
	    !~ /^$cLeader0607Key$/o) {
	    my $lFieldLength;

	    # Isn't this fun!!
	    $lFieldLength = $gControlFields{$lMarcTag}->
			clusters->{
			    substr(
				   &NiceAttribute($lElement->attribute(VALUE)
						  ->value), 0, 1)}->
						      field_length;

	    # Fill the field with "fill" characters before we
	    # start inserting real data.
	    $gPosDefData = $cMarcFill x $lFieldLength;
	}

#        print ":$lStartChar:$lEndChar:";

	# Sometime a range isn't
	if ($lEndChar eq '') {
	    $lEndChar = $lStartChar;
	}

	$lFieldWidth = $lEndChar - $lStartChar + 1;

#        print "$lStartChar:$lEndChar:$lFieldWidth:\n";

	# Get the attribute value
	$lAttributeValue = &NiceAttribute($lElement->attribute(VALUE)->value);

	# If we are a blank or fill character, expand to the field width
	$lAttributeValue = $lAttributeValue x $lFieldWidth
	    if ($lAttributeValue eq $cMarcBlank ||
		$lAttributeValue eq $cMarcFill);

	# Put it where it belongs
	substr($gPosDefData, $lStartChar, $lFieldWidth) =
	    sprintf "%${lFieldWidth}s", $lAttributeValue;

#	print ":$gPosDefData:\n";

    } else {
	# To get here, the tag didn't match the pattern for a
	# positionally-defined cluster

	$gSkipCount++;

	# Record this for posterity
	&Log("Record $gRecordCount has tag,\"$lName\", that does not match positionally-defined element format;  record not converted.");

	# This is an error.
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Error_End);
    }
}

# "PosDef" state end_element handler
# To get here, we have seen the start tag for the wrapper element for a
# positionally-defined field.  The end tags we see may be the spurious
# end tags for the EMPTY cluster fields, in which case we do nothing, or
# may be the real end of the wrapper element, in which case we construct
# the hash entry for the field.
sub PosDef_End {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "PosDef_End:</" . $lElement->name . ">\n";

    # If it's the end tag for the wrapper element...
    if ($lName =~ /^.{4}(\d{3})-..?$/) {
	my ($lMarcTag) = $1;
	my ($lFieldLength) = 0;

#	print "before:$gPosDefData:\n";

	# After blindly inserting data as specified by the numbers
	# in the tags for the clusters, we again set the length of
	# the data just in case some of the cluster tags specified
	# character positions beyond what is allowed.  Of course there
	# are two options for how we determine the length of the
	# field.
	if ($gControlFields{$lMarcTag}->key =~ /^$cLeader0607Key$/o) {
	    if (defined($gControlFields{$lMarcTag}->
			clusters->{$gLeader0607})) {
		$lFieldLength = $gControlFields{$lMarcTag}->
		    clusters->{$gLeader0607}->field_length;
	    } else {
		# Record this, too, for posterity
		&Log("Record $gRecordCount has invalid Leader cp 06-07," .
		     " \"$gLeader0607\", for positionally-defined field" .
		     " \"$lMarcTag\";  record not converted.");
		&MakeErrorState();
		return;
	    }
	} else {
	    $lFieldLength = $gControlFields{$lMarcTag}->
		clusters->{substr($gPosDefData, 0, 1)}->field_length;
	    if (defined($gControlFields{$lMarcTag}->
			clusters->{substr($gPosDefData, 0, 1)})) {
		$lFieldLength = $gControlFields{$lMarcTag}->
		    clusters->{substr($gPosDefData, 0, 1)}->field_length;
	    } else {
		# Record this, too, for posterity
		&Log("Record $gRecordCount has invalid field cp 00, \"".
		     substr($gPosDefData, 0, 1) .
		     "\", for positionally-defined field \"$lMarcTag\";" .
		     "  record not converted.");
		&MakeErrorState();
		return;
	    }
	}

	$gPosDefData = substr $gPosDefData, 0, $lFieldLength;

#	print "after :$gPosDefData:\n";

	# Save the data.  There may be multiple instances of this field
	# number, so the hash value is an anonymous array, and we just
	# add a value to it.
        push(@{$gMarcData{$lMarcTag}}, $gPosDefData);

	# Since this is the end of the positionally-defined field, we're
	# followed by either the start tag of another field or the end
	# tag of the current grouping element.
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&ProcessMarcTag_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&ProcessGroups_End);
    } elsif ($lName =~ /^.{4}(\d{3})-..?-(..)(-..)?$/) {
	# Do nothing if positionally-defined cluster end tag
    } else {
	# Ahah!! An unknown tag

	$gSkipCount++;

	# Record this, too, for posterity
	&Log("Record $gRecordCount has end-tag, \"$lName\", that does not match positionally-defined element format;  record not converted.");

        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Error_End);
      }
}

# "NoIndSf" state end_element handler
# To get here, we saw the start tag for a field with no indicators or
# subfields, so now we are expecting the current element to be the end
# tag.  If it is the end tag, get the character data that we've been
# saving, make it civilized, and save it.
sub NoIndSf_End {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "NoIndSf_End:</" . $lElement->name . ">\n";

    if ($lName =~ /^.{4}(\d{3})$/) {
	# &ConversionRoutine does the entity-character conversion
	push(@{$gMarcData{$1}}, &ConversionRoutine(pop_output));

	# The next tag should be either the start tag for another field
	# or the end tag for the current element group
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&ProcessMarcTag_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&ProcessGroups_End);
    } else {

	$gSkipCount++;

	# Record this for posterity
	&Log("Record $gRecordCount has end-tag, \"$lName\", that does not match record without indicators or subfields element format;  record not converted.");

        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Error_End);
      }
}

# "VariableData" state start_element handler
# To get here, we saw the start element for a data variable field, or a
# previous subfield, so we are expecting the start tag for a data variable
# field.  We may also be the start tag for a subsequent data variable field.
sub VariableData_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "VariableData_Start:<" . $lElement->name . ">\n";

    # Save the character data
    push_output 'string';

    # If we are the start of the wrapper for the data variable field,
    # then save the indicators.
    if ($lName =~ /^.{4}(\d{3})$/) {
	local($lMarcTag) = $1;

	push(@{$gMarcData{$lMarcTag}},
	     (defined($lElement->attributes->{I1}) ?
	      &NiceIndicator($lElement->attribute(I1)->value) :
	      $cMarcBlank) .
	     (defined($lElement->attributes->{I2}) ?
	      &NiceIndicator($lElement->attribute(I2)->value) :
	      $cMarcBlank));
    }

    SGMLSPL::sgml($gInputSGML,
		      'end_element',
		      \&VariableData_End);
}

# "VariableData" start end_element handler
# To get here, we've seen the start tag of the element for a subfield, where
# we saved the character data content of the element with a push_output
# statement
sub VariableData_End {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "VariableData_End:</" . $lElement->name . ">\n";

    # If we are the end tag for a subfield element...
    if ($lName =~ /^.{4}(\d{3})-(.)$/) {
	local($lMarcTag, $lSubfield) = ($1, $2);
	$lSubfield =~ tr/A-Z/a-z/;

	# Append to the last entry in the array that is the value
	# of the hash for the current field.  We append the
	# EOS character, the subfield identifier, and the
	# civilized form of the character data that we saved earlier.
	@{$gMarcData{$lMarcTag}}[$#{$gMarcData{$lMarcTag}}] .=
	    $cMarcEOS . $lSubfield . &ConversionRoutine(pop_output);

	# The next start tag we see is either that of another subfield
	# element or that of a wrapper element for the next data variable
	# field
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&VariableData_Start);

	# If we see an end tag before we see a start tag, it's the end
	# tag for the current group
        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&ProcessGroups_End);
    } else {

	$gSkipCount++;

	# Tell the world (or at least the log file)
	&Log("Record $gRecordCount has end-tag, \"$lName\", that does not match variable data element format;  record not converted.");

        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Error_Start);

        SGMLSPL::sgml($gInputSGML,
			  'end_element',
			  \&Error_End);
      }
}

# "Generic" state start_element handler
# Do nothing and rely on the end_element handler to handle significant
# events.
sub Generic_Start {
    my $lElement = shift;

#    print "Generic_Start:<" . $lElement->name . ">\n";
}

# "Generic" state end_element handler
# Do nothing and rely on the start_element handler to handle significant
# events.
sub Generic_End {
    my $lElement = shift;

#    print "Generic_End:</" . $lElement->name . ">\n";
}

# "Error" start start_element handler
# Stay in the error state until the tag is the start tag for a new MARC
# record.
sub Error_Start {
    my $lElement = shift;
    (my $lName = $lElement->name) =~ tr/A-Z/a-z/;

#    print "Error_Start:<" . $lElement->name . ">\n";

    # Only if the tag is a start tag for a MARC record and it has
    # groups defined in the MARC Description File
    if (defined($gDtdGroups{$lName})) {

	# Save the element name for later use
	$gDoctype = $lName;

	# This is a new record, we just got here unconventionally
	$gRecordCount++;

	# We're back on track, so go to the "Record" state
        SGMLSPL::sgml($gInputSGML,
			  'start_element',
			  \&Record_Start);
    }
}

# "Error" state end_element handler
# Do nothing until the "Error" state start_handler changes the state
sub Error_End {
    my $lElement = shift;

#    print "Error_End:</" . $lElement->name . ">\n";
}

# Set the next state to the Error state
sub SetErrorState {
    $gSkipCount++;

  SGMLSPL::sgml($gInputSGML,
		'start_element',
		\&Error_Start);

  SGMLSPL::sgml($gInputSGML,
		'end_element',
		\&Error_End);
}

# We shouldn't get any SDATA, so warn if we do.
SGMLSPL::sgml($gInputSGML, 'sdata', sub {
    warn "Unknown SDATA: " . $_[0];
});

# Initialize the start_element and end_element handlers
SGMLSPL::sgml($gInputSGML, 'start_element', \&StartProcessing_Start);

SGMLSPL::sgml($gInputSGML, 'end_element', \&Generic_End);

# Pipe the input tag-valid SGML through a Perl script that turns
# "&" into "&#4;" because we are using nsgmls without a DTD and it
# will complain about and delete any entity references that it does
# not recognise.  Since we don't have any entity declarations, it
# would delete every entity reference.  The entity-character conversion
# routine, &ConversionRoutine, expects the \x04 characters, not "&".
# We also turn escape characters, \x1B, into numeric character references
# to get them past nsgmls.  The escape character is used when switching
# character sets.
open (INPUTSGML, "perl -p -e \"s/&/\\&#4;/g; s/\\x1B/\\&#27;/g;\" $gInputFile | nsgmls -wno-valid |");

# Run our state machine on the ESIS from the tag-valid SGML input
SGMLSPL::Process($gInputSGML, INPUTSGML);

# End it all
&Log("End of input file reached.");
&Log("Records processed: $gRecordCount");
&Log("Records converted: $gConvertCount");
&Log("Records skipped: $gSkipCount");
&LogCloseMessage();
close(LOGFILE);

# End of main processing

############################################################
# &BuildMarcRecord()
# Build a MARC record using the data we saved in %gMarcData.
# The field numbers are the keys of %gMarcData, and the values are
# arrays of the data for each occurrence of that field.
sub BuildMarcRecord {
    local($lDirectory) = '';
    local($lMarcRecord) = '';
    local($lRecordLength) = 0;

    # For every field number we found in processing the tag-valid SGML
    # for the MARC record...
    foreach $lMarcTag (sort(keys(%gMarcData))) {
#	print ":$lMarcTag:$gMarcData{$lMarcTag}:\n";
	# For each item in the array for the current field, we make
	# a Directory entry and append it to the string we're building,
	# and append the data to the MARC record that we're building.
	foreach $lMarcData (@{$gMarcData{$lMarcTag}}) {
	    $lDirectory .=
		sprintf "%3s%.4d%.5d",
		    $lMarcTag,
		    length($lMarcData) + 1,
		    length($lMarcRecord);

	    $lMarcRecord .= $lMarcData . $cMarcEOF;
	}
    }

    # Insert the constant fields into the Leader
    substr($gLeaderData, 10, 2) = '22';
    substr($gLeaderData, 20, 4) = '4500';

    # Insert the "Base address of data" field (i.e. where the data starts).
    # This is the length of the Directory, plus 24 character positions for
    # the Leader, plus one character position for the field terminator
    # at the end of the Directory.
    substr($gLeaderData, 12, 5) = sprintf "%.5d",
        length($lDirectory) + 25;

    # Join the Leader, the Directory, the EOF at the end of the Directory,
    # the MARC data that we've been accumulating, and the EOR character
    # to make the complete MARC record.
    $lMarcRecord = $gLeaderData . $lDirectory . $cMarcEOF .
	$lMarcRecord . $cMarcEOR;

    # Find the length of the complete record and put it in the Leader
    # if it's not too long.
    $lRecordLength = length($lMarcRecord);
    if ($lRecordLength <= $cMarcMaxRecordLength) {
	substr($lMarcRecord, 0, 5) = sprintf "%.5d", length($lMarcRecord);
	# Write our record to the output
	print OUTFILE $lMarcRecord;
	$gConvertCount++;
    } else {
	&Log("Record $gRecordCount too long; record not converted.");
    }
}

############################################################
# &NiceIndicator($lAttributeValue)
# Except for the "I1-" or "I2-" prefix, the indicator handling is
# identical, so we lop the prefix.  The words "blank" and "fill" are
# significant, otherwise we just trim the value to being a single
# character and return it.
sub NiceIndicator {
    local($lAttributeValue) = @_;
    local($lData) = $cMarcBlank;

    # Remove the "i1-" or "i2-" from the attribute value
    $lAttributeValue =~ s/^I\d-//i;

    if ($lAttributeValue ne '') {
	if ($lAttributeValue =~ /blank/i) {
	    $lData = $cMarcBlank;
	} elsif ($lAttributeValue =~ /fill/i) {
	    $lData = "$cMarcFill";
	} else {
	    $lData = substr $lAttributeValue, 0, 1;
	}
    }

    $lData;
}

############################################################
# &NiceAttribute($lAttributeValue)
# If the attribute value isn't empty, isn't "blank", and isn't
# "fill", return the attribute value.  "blank" and "fill" are
# significant and get turned into the corresponding characters.
sub NiceAttribute {
    local($lAttributeValue) = @_;
    local($lData) = $cMarcFill;

    if ($lAttributeValue ne '') {
	if ($lAttributeValue =~ /blank/i) {
	    $lData = $cMarcBlank;
	} elsif ($lAttributeValue =~ /fill/i) {
	    $lData = "$cMarcFill";
	} else {
	    $lData = $lAttributeValue;
	}
    }

    $lData;
}

############################################################
# &ConversionRoutine($_)
# This is the default conversion routine that simple converts
# "<" to the "&lt;" entity.  If the user specifies a conversion on
# the command line, then the contents of the conversion specification
# file will be evaluated as a "&ConversionRoutine" subroutine, replacing
# this one.
sub ConversionRoutine {
    local($_) = @_;

    s/\004lt;/</g;
    s/\004amp;/\&/g;

    s/\004/\&/g;

    return($_);
}

############################################################
# &EvalConversion($pConversionFile)
# Read $pConversionFile and make the entity conversion subroutine.
# The conversion subroutine will replace the default "&ConversionRoutine"
# because it is evaluated after the built-in subroutine.
sub EvalConversion {
    local($pConversionFile) = @_;

    # Create a new SGMLSPL object.
    $gEntityMap = new SGMLSPL;

    %gEntityToCharacter = ();

    # Setup the functions to perform for each of the elements in the
    # DTD used for the entity-to-character mapping.  When the conversion
    # file is processed, the anonymous subroutines that are the last
    # argument in each of these subroutine calls will be evaluated as
    # the corresponding start or end tag is processed.

    # For <desc> start tags, save the output
    SGMLSPL::sgml($gEntityMap, '<DESC>', sub {
	push_output 'string';
    });

    # For <desc> end tags, "unsave" the output but don't do anything
    # with it.  In effect, drop the character data content of the
    # <desc> element.
    SGMLSPL::sgml($gEntityMap, '</DESC>', sub {
	pop_output;
    });

    # For <entity> start tags, save the character data content
    SGMLSPL::sgml($gEntityMap, '<ENTITY>', sub {
	push_output 'string';
    });

    # For <entity> end tags, use the saved character data as the key
    # for a hash where the value is the Perl regular expression
    # for the hexadecimal representation of the character numbers
    # in the "hex" attribute of the containing <character> element.
    SGMLSPL::sgml($gEntityMap, '</ENTITY>', sub {
	my $lElement = shift;
	my $lEntityName = pop_output;

	$gEntityToCharacter{$lEntityName} =
	    '\x' . join('\x', split(/\s+/,
				    $lElement->parent->attribute(HEX)->value))
		unless defined($gEntityToCharacter{$lEntityName});
    });

    # Now that we've set up how to process the conversion file,
    # parse the file using nsgmls and process the ESIS output of nsgmls
    open (ENTITYMAP, "nsgmls $cConversionDtdFile $pConversionFile |");
    SGMLSPL::Process($gEntityMap, ENTITYMAP);

    # At this point, we have a hash associating entity names with
    # replacement hexadecimal character sequences.  We now construct
    # the Perl expressions for a subroutine containing a sequence of
    # substitution expressions, then evaluate the string to create a
    # new "&ConversionRoutine" expression.
    $lConversionSubroutine = <<EndOfSubroutineStart;
sub ConversionRoutine {
    local(\$_) = \@_;
    study;
EndOfSubroutineStart

    # The substitution expressions use "\004" (Ctrl-D) where you'd expect
    # "&" because we converted all "&" in the input file to "&#4;"
    # before we parsed because even though we didn't validate, nsgmls
    # would still complain if it saw any entities it wouldn't recognise.
    # Since we don't use a DTD for the input tag-valid SGML, it's best
    # we don't let nsgmls see any entities.
    foreach $lEntity (sort(keys(%gEntityToCharacter))) {

	$lConversionSubroutine .=
	    "    s/\\004$lEntity;/$gEntityToCharacter{$lEntity}/g;\n";
    }

    # We always convert any unconverted "\004" to "&"
    $lConversionSubroutine .= <<EndOfSubroutineEnd;
    s/\\004/\\&/g;

    return(\$_);
}
EndOfSubroutineEnd

    eval($lConversionSubroutine);
    warn $@ if $@;

#    print "Defined\n" if defined(&ConversionRoutine);
}

############################################################
# &FindMarcConvFile($pFile)
# Find the specified file by searching under the directories
# in @INC.  The built-in MARC Description File and entity-to-character
# conversion specification files should be installed in a "Marcconv"
# directory under Perl's "lib" directory.  Not surprisingly, Perl's
# "lib" directory should be in the list of directories in @INC.
sub FindMarcConvFile {
    my($pFile) = shift;

    foreach $lPrefix (@INC) {
	local($lRealFile) = "$lPrefix/Marcconv/$pFile";

	# Return the full pathname of the file if we find it
	return $lRealFile if -f $lRealFile;
    }

    # Warn about not finding the file if we can't find it, and return
    # an empty string.
    warn "Could not locate required file \"$pFile\".\n";
    &Log("Could not locate required file \"$pFile\".");
    return '';
}

############################################################
# Log file and error reporting routines
############################################################

############################################################
# &LogOpenMessage()
# Write the opening message to the log
sub LogOpenMessage {
    &Log("$0 started at " . localtime(time) . "\n");
}

############################################################
# &LogCloseMessage()
# Write the closing message to the log
sub LogCloseMessage {
    &Log("$0 ended at " . localtime(time) . "\n");
}

############################################################
# &Log($pMessage)
# Write $pMessage to the log file
sub Log {
    local($pMessage) = @_;

    chomp($pMessage);

    print LOGFILE $pMessage . "\n";
}

